classdef quaternion
    %quaternion class 
    % A quaternion q corrsponds to ix+jy+kz+r where i^2=j^2=k^2=ijk=-1
    % Note that the order we use for x,y,z,r is compatible with CISST.
    % Note also that methods relating to rotations assume unit quaternions
    % 
    % q  = quaternion(x,y,z,r)  q = ix+jy+kz+r 
    %                           
    % q  = quaternion(q1)       copy constructor
    % q  = quaternion(v)        quaternion(v.x,v.y,v.z,0);
    % q  = quaternion(v,r)      quaternion(v.x,v.y,v.z,r);
    % q  = quaternion(R)        quaternion corresponding to RotMx R
    % q  = quaternion([x,y,z,r]) quaternion (x,y,z,r)
    % q  = quaternion([x,y,z,r]') quaternion (x,y,z,r)
    % 
    % qc = q.conjugate()        qc = quaternion conjugate of q
    % v  = q.VectorPart()       returns vct3 v = vct3(x,y,z)
    % r  = q.RealPart()         returns real part of q
    % v4 = q.vct4()             returns [x,y,z,r]'
    %
    % qu = q.unit()             returns q/q.norm()
    % s  = q.norm()             returns vector norm of q
    % qi = q.Inverse()          returns qi such that qi*q=[0,0,0,1]
    % q  = qa + qb              vector sum of quaternions qa and qb
    % q  = qa - qb              vector difference of quaternions qa and qb
    % qm = -q                   unary minus
    % qp = qa*qb                quaternion product
    % qp = qa*s                 quaternion * scalar
    % qq = qa/qb                qa*qb.Inverse()
    % qq = qa/s                 qa*(1/s)
    % qq = qa\qb                qa.Inverse()*qb
    %
    % qu = quaternion.eye()     quaternion(0,0,0,1)
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor
 
    properties
        x       % imaginary part 
        y
        z
        r       % scalar part
    end
    
    methods
        function q = quaternion(x,y,z,r)
            switch nargin
                case 1
                    switch class(x)
                        case 'quaternion'
                            q = x;
                        case 'vct3'
                            q.x = x(1); q.y = x(2); q.z = x(3); q.r = 0;
                        case 'RotMx'
                            R = x.el;
                            t = trace(R);
                            q.r = 0.5*sqrt(max(0,1+t));
                            q.x = 0.5*sqrt(max(0,1+R(1,1)-R(2,2)-R(3,3)));
                            q.y = 0.5*sqrt(max(0,1-R(1,1)+R(2,2)-R(3,3)));
                            q.z = 0.5*sqrt(max(0,1-R(1,1)-R(2,2)+R(3,3)));
                            if R(3,2)<R(2,3)
                                q.x = -q.x;
                            end
                            if R(1,3)<R(3,1)
                                q.y = -q.y;
                            end
                            if R(2,1)<R(1,2)
                                q.z = -q.z;
                            end                                                      
                        otherwise
                            sz = size(x);
                            if all(sz == [4,1]) || all(sz == [1,4])
                                q.x = x(1); q.y = x(2); q.z = x(3); q.r = x(4);
                            else
                                error('illegal size argument to quaternion constructor');
                            end
                        end
                case 2
                    q.r = y;
                    if isa(x,'vct3') || size(x)==[3,1] || size(x)==[1,3]
                        q.x = x.el(1);
                        q.y = x.el(2);
                        q.z = x.el(3);
                    else
                        error('illegal argument to quaternion constructor');
                    end
                case 4
                    q.x=x; q.y=y; q.z=z; q.r=r;
                otherwise
                    error('illegal argument to quaternion constructor');
            end
        end
        
        function qc = conjugate(q)
            qc = quaternion(-q.x,-q.y,-q.z,q.r);
        end
        
        function v = vct4(q)
            v = [q.x,q.y,q.z,q.r]';
        end
        
        function v = VectorPart(q)
            v = vct3(q.x,q.y,q.z);
        end
        
        function s = RealPart(q)
            s = q.r;
        end
        
        function s = norm(q)
            s = sqrt(q.x^2+q.y^2+q.z^2+q.r^2);
        end
        
        function qu = unit(q)
            qu = q/norm(q);
        end
        
        function qi = Inverse(q)
            D = q.x^2+q.y^2+q.z^2+q.r^2;
            qi = quaternion(-q.x/D,-q.y/D,-q.z/D,q.r/D);
        end
        
        function q = plus(a,b)
            q = quaternion(a.x+b.x,a.y+b.y,a.z+b.z,a.r+b.r);
        end
        
        function q = minus(a,b)
            q = quaternion(a.x-b.x,a.y-b.y,a.z-b.z,a.r-b.r);
        end
        
        function q = uminus(a)
            q = quaternion(-a.x,-a.y,-a.z,-a.r);
        end
        
        function q = product(a,b)
            if isa(b,'quaternion')
                 q = quaternion(a.r*b.x + b.r*a.x + a.y*b.z - a.z*b.y, ...
                                a.r*b.y + b.r*a.y + a.z*b.x - a.x*b.z, ...
                                a.r*b.z + b.r*a.z + a.x*b.y - a.y*b.x, ...
                                a.r*b.r - a.x*b.x - a.y*b.y - a.z*b.z);
            else
                error('bad argument to quaternion multiplication');
            end  
        end
        
        function q = mtimes(a,b)
            if isa(b,'quaternion')
                 q = quaternion(a.r*b.x + b.r*a.x + a.y*b.z - a.z*b.y, ...
                                a.r*b.y + b.r*a.y + a.z*b.x - a.x*b.z, ...
                                a.r*b.z + b.r*a.z + a.x*b.y - a.y*b.x, ...
                                a.r*b.r - a.x*b.x - a.y*b.y - a.z*b.z);
            elseif isscalar(b)
                q = quaternion(a.x*b,a.y*b,a.z*b,a.r*b);              
            else
                error('bad argument to quaternion multiplication');
            end    
        end
    
        function q = mrdivide(a,b)
            if isa(b,'quaternion')
                q = a*b.Inverse();
            elseif isscalar(b)
                q = quaternion(a.x/b,a.y/b,a.z/b,a.r/b);
            else
                error('bad argument to quaternion rdivide');
            end
        end
        
        function q = mldivide(a,b)
            if isa(b,'quaternion')
                q = a.Inverse()*b;
            else
                error('bad argument to quaternion ldivide');
            end
        end
            
                
    end
    
    methods (Static)
    
        function q = eye()
            q = quaternion(0,0,0,1);
        end
        
    end
    
end

